﻿class KSClientSign {
    constructor() {
        this.baseUrl = 'https://localhost:44345'; // Základní URL API
        this.requiredVersion = "1.0.3"
    }

    /**
     * Získá verzi API.
     * @returns {Promise<object|null>} Vrací objekt obsahující informace o verzi API, nebo `null`, pokud je služba nedostupná nebo vrátí chybu.
     */
    async getVersion() {
        try {
            const response = await fetch(`${this.baseUrl}/Sign/GetVersion`, {
                method: 'GET',
                headers: {
                    'Accept': 'application/json',
                },
            });

            if (!response.ok) {
                console.error(`Error fetching API version: HTTP ${response.status}`);
                this.showRadAlert(`Error fetching API version: HTTP ${response.status}`, 'Chyba API');
                return null;
            }

            return await response.json();
        } catch (error) {
            console.error('Error fetching API version:', error.message);
            this.showRadAlert(error.message, 'Chyba API');
            return null;
        }
    }

    /**
    * Testuje, zda je API dostupná, správně funguje a odpovídá požadované verzi.
    * @returns {Promise<boolean>} Vrací `true`, pokud API funguje a verze odpovídá, jinak `false`.
    */
    async testApi() {
        try {
            const version = await this.getVersion();

            if (version) {
                if (version.version === this.requiredVersion) {
                    console.log('API je dostupná a správné verze:', version);
                    return true;
                } else {
                    const message = `Nesprávná verze API. Očekáváno: ${this.requiredVersion}, aktuální: ${version.Version}`;
                    this.showRadAlert(message, 'Chyba API');
                    return false;
                }
            } else {
                this.showRadAlert('Client Service není dostupná nebo vrací chybu.', 'Chyba API');
                return false;
            }
        } catch (error) {
            console.error('Error testing API:', error.message);
            this.showRadAlert('Došlo k neočekávané chybě při testování API.', 'Chyba API');
            return false;
        }
    }

    /**
     * Zobrazí chybovou zprávu v radalert.
     * @param {string} message Zpráva k zobrazení.
     * @param {string} title Název dialogu.
     */
    showRadAlert(message, title) {
        if (window.radalert) {
            window.radalert(message, 330, 150, title);
        } else {
            console.error('RadAlert is not available. Showing fallback alert.');
            alert(`${title}: ${message}`);
        }
    }

    /**
     * Získá seznam platných certifikátů.
     * @returns {Promise<object[]>} Vrací pole certifikátů.
     */
    async getCertificates() {
        try {
            const response = await fetch(`${this.baseUrl}/Sign/GetCertificates`, {
                method: 'GET',
                headers: {
                    'Accept': 'application/json',
                },
            });
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return await response.json();
        } catch (error) {
            console.error('Error fetching certificates:', error);
            throw error;
        }
    }

    /**
    * Podepíše PDF dokument a vrátí objekt s podepsaným souborem v Base64 a ID.
    * @param {string} base64Pdf - Base64 PDF dokumentu.
    * @param {string} signatureFieldName - Název podpisového pole v dokumentu.
    * @param {string} thumbprint - Otisk certifikátu pro podepisování.
    * @param {number} id - Identifikátor dokumentu.
    * @param {string} tsUrl - URL pro časové razítko.
    * @param {string} tsUser - Uživatel pro časové razítko.
    * @param {string} tsPass - Heslo pro časové razítko.
    * @returns {Promise<{ file: string, id: number }>} Vrací objekt s podepsaným PDF v Base64 a ID.
    */
    async signPdf(base64Pdf, signatureFieldName, thumbprint, id, tsUrl, tsUser, tsPass) {
        if (thumbprint) {
            try {
                const payload = {
                    Id: id,
                    PdfBytes: base64Pdf,
                    SignatureFieldName: signatureFieldName,
                    Thumbprint: thumbprint,
                    TimeStampUrl: tsUrl,
                    TimeStampUser: tsUser,
                    TimeStampPass: tsPass,
                };

                const response = await fetch(`${this.baseUrl}/Sign/SignPdf`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(payload),
                });

                if (!response.ok) {
                    const errorText = await response.text();
                    this.showRadAlert(errorText, 'Chyba API');
                    throw new Error(`HTTP error! status: ${response.status} - ${errorText}`);
                }

                return await response.json(); // Vrací objekt { file: "base64", id: number }

            } catch (error) {
                console.error('Error signing PDF:', error);
                throw error;
            }
        }
    }
}